// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)
    based on code by Marc Shapiro (shapiro@sor.inria.fr)

This file is part of the GNU C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef __GNUG__
#pragma implementation
#endif
#include "COb.XPlex.h"


CObXPlex:: CObXPlex()
{
  lo = fnc = 0;
  csize = DEFAULT_INITIAL_CAPACITY;
  COb* data = new COb[csize];
  set_cache(new CObIChunk(data,  lo, lo, fnc, lo+csize));
  hd = ch;
}

CObXPlex:: CObXPlex(int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = 0;
  if (chunksize > 0)
  {
    csize = chunksize;
    COb* data = new COb[csize];
    set_cache(new CObIChunk(data,  lo, lo, fnc, csize));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    COb* data = new COb[csize];
    set_cache(new CObIChunk(data,  chunksize, lo, fnc, fnc));
    hd = ch;
  }
}


CObXPlex:: CObXPlex(int l, int chunksize)
{
  if (chunksize == 0) error("invalid constructor specification");
  lo = fnc = l;
  if (chunksize > 0)
  {
    csize = chunksize;
    COb* data = new COb[csize];
    set_cache(new CObIChunk(data,  lo, lo, fnc, csize+lo));
    hd = ch;
  }
  else
  {
    csize = -chunksize;
    COb* data = new COb[csize];
    set_cache(new CObIChunk(data,  chunksize+lo, lo, fnc, fnc));
    hd = ch;
  }
}

void CObXPlex::make_initial_chunks(int up)
{
  int need = fnc - lo;
  hd = 0;
  if (up)
  {
    int l = lo;
    do
    {
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      COb* data = new COb [csize];
      CObIChunk* h = new CObIChunk(data,  l, l, l+sz, l+csize);
      if (hd != 0)
        h->link_to_next(hd);
      else
        hd = h;
      l += sz;
      need -= sz;
    } while (need > 0);
  }
  else
  {
    int hi = fnc;
    do
    {
      int sz;
      if (need >= csize)
        sz = csize;
      else
        sz = need;
      COb* data = new COb [csize];
      CObIChunk* h = new CObIChunk(data,  hi-csize, hi-sz, hi, hi);
      if (hd != 0)
        h->link_to_next(hd);
      hd = h;
      hi -= sz;
      need -= sz;
    } while (need > 0);
  }
  set_cache(hd);
}

CObXPlex:: CObXPlex(int l, int hi, const COb  initval, int chunksize)
{
  lo = l;
  fnc = hi + 1;
  if (chunksize == 0)
  {
    csize = fnc - l;
    make_initial_chunks(1);
  }
  else if (chunksize < 0)
  {
    csize = -chunksize;
    make_initial_chunks(0);
  }
  else
  {
    csize = chunksize;
    make_initial_chunks(1);
  }
  fill(initval);
}

CObXPlex::CObXPlex(const CObXPlex& a)
{
  lo = a.lo;
  fnc = a.fnc;
  csize = a.csize;
  make_initial_chunks();
  for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
}

void CObXPlex::operator= (const CObXPlex& a)
{
  if (&a != this)
  {
    invalidate();
    lo = a.lo;
    fnc = a.fnc;
    csize = a.csize;
    make_initial_chunks();
    for (int i = a.low(); i < a.fence(); a.next(i)) (*this)[i] = a[i];
  }
}


void CObXPlex::cache(int idx) const
{
  const CObIChunk* tail = tl();
  const CObIChunk* t = ch;
  while (idx >= t->fence_index())
  {
    if (t == tail) index_error();
    t = (t->next());
  }
  while (idx < t->low_index())
  {
    if (t == hd) index_error();
    t = (t->prev());
  }
  set_cache(t);
}


void CObXPlex::cache(const COb* p) const
{
  const CObIChunk* old = ch;
  const CObIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) index_error();
  }
  set_cache(t);
}

int CObXPlex::owns(Pix px) const
{
  COb* p = (COb*)px;
  const CObIChunk* old = ch;
  const CObIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old) { set_cache(t); return 0; }
  }
  set_cache(t);
  return 1;
}


COb* CObXPlex::dosucc(const COb* p) const
{
  if (p == 0) return 0;
  const CObIChunk* old = ch;
  const CObIChunk* t = ch;
 
  while (!t->actual_pointer(p))
  {
    t = (t->next());
    if (t == old)  return 0;
  }
  int i = t->index_of(p) + 1;
  if (i >= fnc) return 0;
  if (i >= t->fence_index()) t = (t->next());
  set_cache(t);
  return (t->pointer_to(i));
}

COb* CObXPlex::dopred(const COb* p) const
{
  if (p == 0) return 0;
  const CObIChunk* old = ch;
  const CObIChunk* t = ch;
  while (!t->actual_pointer(p))
  {
    t = (t->prev());
    if (t == old) return 0;
  }
  int i = t->index_of(p) - 1;
  if (i < lo) return 0;
  if (i < t->low_index()) t = (t->prev());
  set_cache(t);
  return (t->pointer_to(i));
}


int CObXPlex::add_high(const COb  elem)
{
  CObIChunk* t = tl();
  if (!t->can_grow_high())
  {
    if (t->CObIChunk::empty() && one_chunk())
      t->clear(fnc);
    else
    {
      COb* data = new COb [csize];
      t = (new CObIChunk(data,  fnc, fnc, fnc,fnc+csize));
      t->link_to_prev(tl());
    }
  }
  *((t->CObIChunk::grow_high())) = elem;
  set_cache(t);
  return fnc++;
}

int CObXPlex::del_high ()
{
  if (empty()) empty_error();
  CObIChunk* t = tl();
  t->CObIChunk::shrink_high();
  if (t->CObIChunk::empty() && !one_chunk())
  {
    CObIChunk* pred = t->prev();
    del_chunk(t);
    t = pred;
  }
  set_cache(t);
  return --fnc - 1;
}

int CObXPlex::add_low (const COb  elem)
{
  CObIChunk* t = hd;
  if (!t->can_grow_low())
  {
    if (t->CObIChunk::empty() && one_chunk())
      t->cleardown(lo);
    else
    {
      COb* data = new COb [csize];
      hd = new CObIChunk(data,  lo-csize, lo, lo, lo);
      hd->link_to_next(t);
      t = hd;
    }
  }
  *((t->CObIChunk::grow_low())) = elem;
  set_cache(t);
  return --lo;
}


int CObXPlex::del_low ()
{
  if (empty()) empty_error();
  CObIChunk* t = hd;
  t->CObIChunk::shrink_low();
  if (t->CObIChunk::empty() && !one_chunk())
  {
    hd = t->next();
    del_chunk(t);
    t = hd;
  }
  set_cache(t);
  return ++lo;
}

void CObXPlex::reverse()
{
  COb tmp;
  int l = lo;
  int h = fnc - 1;
  CObIChunk* loch = hd;
  CObIChunk* hich = tl();
  while (l < h)
  {
    COb* lptr = loch->pointer_to(l);
    COb* hptr = hich->pointer_to(h);
    tmp = *lptr;
    *lptr = *hptr;
    *hptr = tmp;
    if (++l >= loch->fence_index()) loch = loch->next();
    if (--h < hich->low_index()) hich = hich->prev();
  }
}

void CObXPlex::fill(const COb  x)
{
  for (int i = lo; i < fnc; ++i) (*this)[i] = x;
}

void CObXPlex::fill(const COb  x, int l, int hi)
{
  for (int i = l; i <= hi; ++i) (*this)[i] = x;
}


void CObXPlex::clear()
{
  if (fnc != lo)
  {
    CObIChunk* t = tl();
    while (t != hd)
    {
      CObIChunk* prv = t->prev();
      del_chunk(t);
      t = prv;
    }
    t->CObIChunk::clear(lo);
    set_cache(t);
    fnc = lo;
  }
}


int CObXPlex::OK () const
{
  int v = hd != 0 && ch != 0;     // at least one chunk

  v &= fnc == tl()->fence_index();// last chunk fence == plex fence
  v &= lo == ((hd))->CObIChunk::low_index();    // first lo == plex lo

// loop for others:
  int found_ch = 0;                   // to make sure ch is in list;
  const CObIChunk* t = (hd);
  for (;;)
  {
    if (t == ch) ++found_ch;
    v &= t->CObIChunk::OK();              // each chunk is OK
    if (t == tl())
      break;
    else                              // and has indices contiguous to succ
    {
      v &= t->top_index() == t->next()->base_index();
      if (t != hd)                  // internal chunks full
      {
        v &= !t->empty();
        v &= !t->can_grow_low();
        v &= !t->can_grow_high();
      }
      t = t->next();
    }
  }
  v &= found_ch == 1;
  if (!v) error("invariant failure");
  return v;
}
